﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;

namespace VA.PPMS.Context
{
    public class PpmsHelper : IPpmsHelper
    {
        private readonly IPpmsContextHelper _ppmsContextHelper;

        public PpmsHelper(IPpmsContextHelper ppmsContextHelper)
        {
            _ppmsContextHelper = ppmsContextHelper;
        }

        public SetStateRequest SetEntityStatus<T>(T entity, int state, int statuscode)
        {
            try
            {
                if (entity != null)
                {
                    var crmEntity = entity as Entity;
                    if (crmEntity != null)
                    {
                        var setStateRequest = new SetStateRequest
                        {
                            EntityMoniker = new EntityReference(typeof(T).Name.ToLower(), crmEntity.Id),
                            State = new OptionSetValue(state),
                            Status = new OptionSetValue(statuscode)
                        };
                        return setStateRequest;
                    }
                }
            }
            catch (Exception ex)
            {
                throw new InvalidOperationException("Exception occurred when setting new entity status.", ex);
            }
            return null;
        }

        public async Task<ppms_batch> GetBatch(DasMessage message)
        {
            using (var context = await _ppmsContextHelper.GetContextAsync())
            {
                var batch = await GetBatch(context, message);
                if (batch != null)
                {
                    context.LoadProperty(batch, new Relationship("ppms_batch_batchdetail_batch"));

                    if (batch.ppms_batch_batchdetail_batch != null)
                    {
                        foreach (var item in batch.ppms_batch_batchdetail_batch)
                        {
                            context.LoadProperty(item, "ppms_batchdetail_batchdetailresult");
                        }
                    }
                }

                return batch;
            }
        }

        public async Task<ppms_batch> GetBatch(PpmsContext context, DasMessage message)
        {
            await Task.Run(() => {});

            return context.ppms_batchSet.FirstOrDefault(x => x.ppms_conversationid == message.ConversationId);
        }

        public async Task<bool> CreateBatch(DasMessage dasMessage, string updateMessage, int statusCode)
        {
            try
            {
                // Get context
                using (var context = await _ppmsContextHelper.GetContextAsync())
                {
                    var networkGuid = await GetNetworkIdByShorthand(dasMessage.SenderId);
                    // Convert network ID to a GUID
                    if (!networkGuid.HasValue)
                        return false;

                    // Lookup network
                    var network = context.ppms_vaprovidernetworkSet.FirstOrDefault(x => x.Id == networkGuid.Value);

                    if (network == null) throw new InvalidOperationException("The network referenced is invalid.");

                    // Create Batch entity
                    var batch = new ppms_batch
                    {
                        ppms_conversationid = dasMessage.ConversationId,
                        ppms_transactionid = dasMessage.TransactionId,
                        ppms_vaprovidernetwork_batch_network = network,
                        ppms_history = updateMessage,
                        ppms_starttime = DateTime.Now
                    };
                    context.AddObject(batch);

                    // Save changes
                    var saveChanges = context.SaveChanges();

                    if (saveChanges.HasError)
                    {
                        var sb = new StringBuilder();
                        foreach (var saveChange in saveChanges)
                        {
                            if (saveChange.Error != null)
                            {
                                sb.AppendFormat("Error: {0}\n", saveChange.Error.Message);
                            }
                        }
                        throw new InvalidOperationException("Failure occurred while saving Batch records: " + sb);
                    }

                    // Change status
                    var request = SetEntityStatus(batch, (int)ppms_batchState.Active, statusCode);
                    context.Execute(request);

                    return true;
                }
            }
            catch (Exception ex)
            {
                throw new InvalidOperationException("Failure occurred while saving Batch records: " + ex.Message);
            }
        }

        public async Task<bool> UpdateBatch(DasMessage dasMessage, string updateMessage, int statusCode)
        {
            // Get context
            using (var context = await _ppmsContextHelper.GetContextAsync())
            {
                try
                {
                    // Lookup batch record
                    var batch = await GetBatch(context, dasMessage);

                    if (batch == null) throw new InvalidOperationException("The conversation record does not exist.");

                    // Update Batch entity
                    batch.ppms_history = FormatHistory(batch.ppms_history, updateMessage);
                    context.UpdateObject(batch);

                    // Save changes
                    var saveChanges = context.SaveChanges();

                    if (saveChanges.HasError)
                    {
                        var sb = new StringBuilder();
                        foreach (var saveChange in saveChanges)
                        {
                            if (saveChange.Error != null)
                            {
                                sb.AppendFormat("Error: {0}\n", saveChange.Error.Message);
                            }
                        }
                        throw new InvalidOperationException("Failure occurred while saving Batch record: " + sb);
                    }

                    // Change status
                    var request = SetEntityStatus(batch, (int)ppms_batchState.Active, statusCode);
                    context.Execute(request);

                    return true;
                }
                catch (Exception ex)
                {
                    throw new PpmsServiceException("Exception occurred while saving Batch record: " + ex.Message);
                }
            }
        }

        public async Task IncrementBatchStatus(DasMessage message, string activityMessage)
        {
            var batch = await GetBatch(message);
            if (batch != null)
            {
                var newStatus = NextBatchStatusCode(batch.StatusCode.Value);
                if (newStatus.HasValue)
                {
                    await UpdateBatch(message, activityMessage, (int)newStatus);
                }
            }
        }

        public async Task<Guid?> GetNetworkIdByShorthand(string shorthand)
        {
            using (var context = await _ppmsContextHelper.GetContextAsync())
            {
                if (!string.IsNullOrEmpty(shorthand))
                {
                    var network = context.ppms_vaprovidernetworkSet.FirstOrDefault(x => x.ppms_shorthand == shorthand.Trim());
                    if (network != null)
                    {
                        return network.ppms_vaprovidernetworkId;
                    }
                }

                return null;
            }
        }

        private static string FormatHistory(string previous, string current)
        {
            var newEntry = $"{DateTime.Now} - {current}";

            var sb = new StringBuilder();
            sb.Append(previous);
            sb.AppendLine(newEntry);
            var result = sb.ToString();
            return result.Substring(Math.Max(0, result.Length - 6000));
        }

        private static ppms_batch_StatusCode? NextBatchStatusCode(int status)
        {
            ppms_batch_StatusCode? currentStatus = (ppms_batch_StatusCode)status;
            //if (!currentStatus.HasValue) return null;

            switch (currentStatus)
            {
                case ppms_batch_StatusCode.DataReceived:
                    return ppms_batch_StatusCode.DataRequested;
                case ppms_batch_StatusCode.DataRequested:
                    return ppms_batch_StatusCode.DataRequestValidated;
                case ppms_batch_StatusCode.DataRequestValidated:
                    return ppms_batch_StatusCode.DataScanSuccessful;
                case ppms_batch_StatusCode.DataScanSuccessful:
                    return ppms_batch_StatusCode.ReceiverNotified;
                case ppms_batch_StatusCode.ReceiverNotified:
                    return ppms_batch_StatusCode.RequestInitiated;
                case ppms_batch_StatusCode.RequestInitiated:
                    return ppms_batch_StatusCode.Processing;
                case ppms_batch_StatusCode.ResponseDataRequested:
                    return ppms_batch_StatusCode.ResponseInitiated;
                case ppms_batch_StatusCode.ResponseInitiated:
                    return ppms_batch_StatusCode.DataRequestValidated;
                default:
                    return null;
            }
        }
    }
}